//
//  v002ScreenCapturePlugIn.m
//  v002ScreenCapturePlugIn
//
//  Created by vade on 6/17/08.
//  Copyright (c) 2008 'vade'. All rights reserved.
//

/* It's highly recommended to use CGL macros instead of changing the current context for plug-ins that perform OpenGL rendering */
#import <OpenGL/CGLMacro.h>

#import "v002ScreenCapturePlugIn.h"

#define	kQCPlugIn_Name				@"v002ScreenCapture"
#define	kQCPlugIn_Description		@"Allows you to selectively return a sub portion of the desktop, and use it as a texture."
#define kQCPlugIn_Copyright			@"Anton Marini \"vade\" - 2008 - with much help and thanks to Maxx Rupp and #macdev on freenode"

@implementation v002ScreenCapturePlugIn

#pragma mark -
#pragma mark Plugin Callbacks
#pragma mark -

static GLuint  _CreateWindowTexture(CGLContextObj internalContext, CGLContextObj mainDrawingContext, NSInteger width, NSInteger height, NSInteger originX, NSInteger originY,NSRect bounds)
{	
	CGLContextObj cgl_ctx = internalContext;
	
	// Thread lock OpenGL Context
	CGLLockContext(cgl_ctx);  
	
	GLuint mTextureName;
	GLenum theError = GL_NO_ERROR;
	
	// set up our texture storage for copying
	glPixelStorei(GL_UNPACK_ALIGNMENT, 4);
	glPixelStorei(GL_PACK_ROW_LENGTH, 0);
    glPixelStorei(GL_PACK_SKIP_ROWS, 0);
    glPixelStorei(GL_PACK_SKIP_PIXELS, 0);
	
	// Create and configure the texture - rectangluar coords
	glGenTextures(1, &mTextureName);
	glBindTexture(GL_TEXTURE_RECTANGLE_ARB, mTextureName);
	glTexParameteri(GL_TEXTURE_RECTANGLE_ARB, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
	
	// define our texture - we're allowd to supply a null pointer since we are letting the GPU handle texture storage.
	glTexImage2D(GL_TEXTURE_RECTANGLE_ARB,0,GL_RGBA, width,height,0,GL_BGRA, GL_UNSIGNED_INT_8_8_8_8_REV, NULL);
	
	// read from the front buffer
	glReadBuffer(GL_FRONT);
//	glBindTexture(GL_TEXTURE_RECTANGLE_ARB, mTextureName);
	
	// copy contents of a portion of the buffer to our texture
	glCopyTexSubImage2D(GL_TEXTURE_RECTANGLE_ARB, 0, 0, 0, originX, originY, width, height);
	
	// fin
	glFlush();	
	
	// Thread unlock
	CGLUnlockContext(cgl_ctx); 
	
	//Check for OpenGL errors
	theError = glGetError();
	if(theError) {
		NSLog(@"v002ScreenCapture: OpenGL texture creation failed (error 0x%04X)", theError);
		CGLUnlockContext(cgl_ctx); // Thread unlock
		return 0;
	}
	
	return mTextureName;
}


#pragma mark -
#pragma mark Private Window related plugin routines
#pragma mark -

- (BOOL) initFullScreenGLContext:(id<QCPlugInContext>)context
{	
	if([context CGLContextObj] == NULL)
	{
		NSLog(@"v002ScreenCapture: Odd, we seem to have no QC Plugin Context... failing");
		return NO;
	}
	
	// we need to create our fullscreen GL Context for screen capturing.
	// this context is different than QC Plugin hosts GL context, which we have to use for drawing.
	// we set fullscreenCaptureContext to share resources with our QC Context
	// however to ensure we can share resources, we copy its pixel format, and use it to create ours.
	
	//NSLog(@"called initFullScreenGLContext");
	
	// get the display mask used by the host renderer
	/*	GLint mask;
	 CGLPixelFormatObj obj = CGLGetPixelFormat([context CGLContextObj]);
	 CGLDescribePixelFormat(obj, 0, kCGLPFADisplayMask, &mask);
	 */	
	
	// the commented out pixel format attributes are left for reference. We use ALLRenderers now, but may cause issues?
	CGLPixelFormatAttribute attributes[] = {
												kCGLPFAAllRenderers,
												//	kCGLPFADoubleBuffer,	// works
												//	kCGLPFAColorSize, 32,	// works
												//	kCGLPFAAlphaSize, 8,	// works
												//	kCGLPFADepthSize, 24,	// works
												kCGLPFAFullScreen,		// required
												kCGLPFADisplayMask, CGDisplayIDToOpenGLDisplayMask(kCGDirectMainDisplay),	// required
												//	kCGLPFADisplayMask, mask,
												0,
											};
	
	CGLPixelFormatObj pixelFormatObj2;
	GLint numPixelFormats = 0;
	
	CGLError err = CGLChoosePixelFormat(attributes, &pixelFormatObj2, &numPixelFormats);
	
	if(err != 0) // kCGLNoError
	{
		NSLog(@"v002ScreenCapture: CGL PixelFormat Error: %@", [NSString stringWithFormat: @"%s", CGLErrorString(err)] );
		NSLog(@"v002ScreenCapture: Failed creating pixel format");
		return NO;
	}
	
	err = CGLCreateContext(pixelFormatObj2, [context CGLContextObj], &fullScreenCaptureContext);
	
	if(err != 0) // kCGLNoError
	{
		NSLog(@"v002ScreenCapture: CGL Context Error: %@", [NSString stringWithFormat: @"%s", CGLErrorString(err)] );
		NSLog(@"v002ScreenCapture: Failed creating full screen GL context");
		return NO;
	}
	
	CGLDestroyPixelFormat(pixelFormatObj2);
	
	// set it full screen	
	err = CGLSetFullScreen(fullScreenCaptureContext);
	if(err != 0) // kCGLNoError
	{
		NSLog(@"v002ScreenCapture: CGL Context Error: %@", [NSString stringWithFormat: @"%s", CGLErrorString(err)] );
		NSLog(@"v002ScreenCapture: Failed going full screen on our GL context");
		return NO;
	}
	
	//NSLog(@"created our full screen GL context");
	
	return YES;
}

#pragma mark -
#pragma mark Required plugin routines
#pragma mark -

/*
 Here you need to declare the input / output properties as dynamic as Quartz Composer will handle their implementation
 @dynamic inputFoo, outputBar;
 */

@dynamic inputOriginX, inputOriginY, inputHeight, inputWidth, outputWindowImage;

+ (NSDictionary*) attributes
{
	/*
	 Return a dictionary of attributes describing the plug-in (QCPlugInAttributeNameKey, QCPlugInAttributeDescriptionKey...).
	 */
	
	return [NSDictionary dictionaryWithObjectsAndKeys:kQCPlugIn_Name, QCPlugInAttributeNameKey, 
			kQCPlugIn_Description, QCPlugInAttributeDescriptionKey, 
			kQCPlugIn_Copyright, QCPlugInAttributeCopyrightKey, nil];
}

+ (NSDictionary*) attributesForPropertyPortWithKey:(NSString*)key
{
	/*
	 Specify the optional attributes for property based ports (QCPortAttributeNameKey, QCPortAttributeDefaultValueKey...).
	 */
	
	
	if([key isEqualToString:@"inputOriginX"])
	{
		return [NSDictionary dictionaryWithObjectsAndKeys:@"Origin (X)", QCPortAttributeNameKey,
				[NSNumber numberWithFloat:0.0], QCPortAttributeMinimumValueKey,
				[NSNumber numberWithFloat:0.0], QCPortAttributeDefaultValueKey,
				[NSNumber numberWithFloat:[[NSScreen mainScreen] frame].size.width], QCPortAttributeMaximumValueKey,
				nil];
		
	}
	
	if([key isEqualToString:@"inputOriginY"])
	{
		return [NSDictionary dictionaryWithObjectsAndKeys:@"Origin (Y)", QCPortAttributeNameKey,
				[NSNumber numberWithFloat:0.0], QCPortAttributeMinimumValueKey,
				[NSNumber numberWithFloat:0.0], QCPortAttributeDefaultValueKey,
				[NSNumber numberWithFloat:[[NSScreen mainScreen] frame].size.height], QCPortAttributeMaximumValueKey,
				nil];
	}
	
	if([key isEqualToString:@"inputWidth"])
	{
		return [NSDictionary dictionaryWithObjectsAndKeys:@"Capture Width", QCPortAttributeNameKey,
				[NSNumber numberWithDouble:320], QCPortAttributeDefaultValueKey,
				[NSNumber numberWithDouble:1], QCPortAttributeMinimumValueKey,
				[NSNumber numberWithFloat:[[NSScreen mainScreen] frame].size.width], QCPortAttributeMaximumValueKey,
				nil];
	}
	
	if([key isEqualToString:@"inputHeight"])
	{
		return [NSDictionary dictionaryWithObjectsAndKeys:@"Capture Height", QCPortAttributeNameKey,
				[NSNumber numberWithDouble:240.0], QCPortAttributeDefaultValueKey,
				[NSNumber numberWithDouble:1], QCPortAttributeMinimumValueKey,
				[NSNumber numberWithFloat:[[NSScreen mainScreen] frame].size.height], QCPortAttributeMaximumValueKey,
				nil];
	}
	
	if([key isEqualToString:@"outputWindowImage"])
	{
		return [NSDictionary dictionaryWithObjectsAndKeys:@"Window Image", QCPortAttributeNameKey,nil];
	}
	
	return nil;
}

+ (QCPlugInExecutionMode) executionMode
{
	/*
	 Return the execution mode of the plug-in: kQCPlugInExecutionModeProvider, kQCPlugInExecutionModeProcessor, or kQCPlugInExecutionModeConsumer.
	 */
	return kQCPlugInExecutionModeProvider;
}

+ (QCPlugInTimeMode) timeMode
{
	/*
	 Return the time dependency mode of the plug-in: kQCPlugInTimeModeNone, kQCPlugInTimeModeIdle or kQCPlugInTimeModeTimeBase.
	 */
	
	return kQCPlugInTimeModeNone;
}

- (id) init
{
	if(self = [super init])
	{
	}
	
	return self;
}

- (void) finalize
{
	/*
	 Release any non garbage collected resources created in -init.
	 */
	
	[super finalize];
}

- (void) dealloc
{
	/*
	 Release any resources created in -init.
	 */
	[super dealloc];
}


@end



@implementation v002ScreenCapturePlugIn (Execution)

- (BOOL) startExecution:(id<QCPlugInContext>)context
{
	/*
	 Called by Quartz Composer when rendering of the composition starts: perform any required setup for the plug-in.
	 Return NO in case of fatal failure (this will prevent rendering of the composition to start).
	 */
	
	if(![self initFullScreenGLContext:context])
	{	
		NSLog(@"v002ScreenCapture: startExecution: failed initing full screen context");
		return NO;
	}
	
	return YES;
}

- (void) enableExecution:(id<QCPlugInContext>)context
{
	/*
	 Called by Quartz Composer when the plug-in instance starts being used by Quartz Composer.
	 */
}

- (BOOL) execute:(id<QCPlugInContext>)context atTime:(NSTimeInterval)time withArguments:(NSDictionary*)arguments
{	
	
	id provider;
	
	provider = [[windowImageProviderGL alloc] initWithFullScreenContext:fullScreenCaptureContext 
																 origin:NSMakePoint(self.inputOriginX, self.inputOriginY) 
																   size:NSMakeSize(self.inputWidth ,self.inputHeight)];
	
	self.outputWindowImage = provider;
	[provider release];
	
	return YES;
}

- (void) disableExecution:(id<QCPlugInContext>)context
{
}

- (void) stopExecution:(id<QCPlugInContext>)context
{
}

@end

#pragma mark -
#pragma mark Window Image QCPlugInOutputImageProvider
#pragma mark -

@implementation windowImageProviderGL

- (id) initWithFullScreenContext:(CGLContextObj)context origin:(NSPoint) origin size:(NSSize)size
{	
	if(self = [super init])
	{
		fullScreenCaptureContext = context;
		
		// window position within the main screen/front buffer
		width = (GLint) size.width;
		height = (GLint) size.height;
		originX = (GLint) origin.x;
		originY = (GLint) origin.y;
		
		return self;
	}
	return nil;
}

- (void) dealloc
{	
	[super dealloc];
}


- (GLuint) copyRenderedTextureForCGLContext:(CGLContextObj)cgl_ctx pixelFormat:(NSString*)format bounds:(NSRect)bounds isFlipped:(BOOL*)flipped
{
	return _CreateWindowTexture(fullScreenCaptureContext, cgl_ctx, width, height, originX, originY, bounds);
	
	//	use this to debug in QC editor, it will make it not crash. because, QC is a pain in my ass
	//	return 0;
}

- (void) releaseRenderedTexture:(GLuint)name forCGLContext:(CGLContextObj)cgl_ctx
{
	glDeleteTextures(1, &name);
}

- (NSRect) imageBounds
{
	return NSMakeRect(0.0, 0.0, width, height);
}

- (CGColorSpaceRef) imageColorSpace
{
	return CGColorSpaceCreateWithName(kCGColorSpaceGenericRGB);
}

- (NSArray*) supportedRenderedTexturePixelFormats
{
	//NSLog(@"called supportedRenderedTexturePixelFormats");
#if __BIG_ENDIAN__
	return [NSArray arrayWithObjects:QCPlugInPixelFormatARGB8,QCPlugInPixelFormatRGBAf,nil];
#else
	return [NSArray arrayWithObjects:QCPlugInPixelFormatBGRA8,QCPlugInPixelFormatRGBAf,nil];
#endif
}

@end

